unit KM_Utils;
{$I KaM_Remake.inc}
interface
uses Classes, DateUtils, Math, SysUtils, KM_Defaults, KM_Points
  {$IFDEF MSWindows}
  ,MMSystem //Required for TimeGet which is defined locally because this unit must NOT know about KromUtils as it is not Linux compatible (and this unit is used in Linux dedicated servers)
  {$ENDIF}
  ;

  function KMGetCursorDirection(X,Y: integer): TKMDirection;

  function GetPositionInGroup2(OriginX, OriginY: Word; aDir: TKMDirection; aIndex, aUnitPerRow: Word; MapX, MapY: Word; out aTargetCanBeReached: Boolean): TKMPoint;
  function GetPositionFromIndex(aOrigin: TKMPoint; aIndex: Byte): TKMPointI;

  function FixDelim(const aString: string): string;

  procedure ConvertRGB2HSB(aR, aG, aB: Integer; out oH, oS, oB: Single);
  procedure ConvertHSB2RGB(aHue, aSat, aBri: Single; out R, G, B: Byte);
  function ApplyBrightness(aColor: Cardinal; aBrightness: Byte): Cardinal;
  function GetPingColor(aPing: Word): Cardinal;
  function GetFPSColor(aFPS: Word): Cardinal;
  function FlagColorToTextColor(aColor: Cardinal): Cardinal;
  function TimeToString(aTime: TDateTime): string;
  function WrapColor(aText: UnicodeString; aColor: Cardinal): UnicodeString;
  function WrapColorA(aText: AnsiString; aColor: Cardinal): AnsiString;
  function StripColor(aText: UnicodeString): UnicodeString;

  procedure ParseDelimited(const SL: TStringList; const Value: string; const Delimiter: string);

  procedure SetKaMSeed(aSeed:integer);
  function GetKaMSeed:integer;
  function KaMRandom:extended; overload;
  function KaMRandom(aMax:integer):integer; overload;
  function KaMRandomS(Range_Both_Directions:integer):integer; overload;
  function KaMRandomS(Range_Both_Directions:single):single; overload;

  function TimeGet: Cardinal;
  function GetTimeSince(aTime: Cardinal): Cardinal;

  function MapSizeText(X,Y: Word): string;

  //Taken from KromUtils to reduce dependancies (required so the dedicated server compiles on Linux without using Controls)
  procedure KMSwapInt(var A,B:byte); overload;
  procedure KMSwapInt(var A,B:shortint); overload;
  procedure KMSwapInt(var A,B:smallint); overload;
  procedure KMSwapInt(var A,B:word); overload;
  procedure KMSwapInt(var A,B:integer); overload;
  procedure KMSwapInt(var A,B:cardinal); overload;

implementation


var
  fKaMSeed: Integer;


//Taken from KromUtils to reduce dependancies (required so the dedicated server compiles on Linux without using Controls)
function GetLength(A, B: Single): Single;
begin
  Result := Sqrt(Sqr(A) + Sqr(B));
end;

procedure KMSwapInt(var A, B: byte);
var
  S: byte;
begin
  S := A; A := B; B := S;
end;

procedure KMSwapInt(var A, B: shortint);
var
  S: shortint;
begin
  S := A; A := B; B := S;
end;

procedure KMSwapInt(var A,B:smallint);
var s:smallint;
begin
  s:=A; A:=B; B:=s;
end;

procedure KMSwapInt(var A,B:word);
var s:word;
begin
  s:=A; A:=B; B:=s;
end;

procedure KMSwapInt(var A,B:integer);
var s:integer;
begin
  s:=A; A:=B; B:=s;
end;

procedure KMSwapInt(var A,B:cardinal);
var s:cardinal;
begin
  s:=A; A:=B; B:=s;
end;


//This unit must not know about KromUtils because it is used by the Linux Dedicated servers
//and KromUtils is not Linux compatible. Therefore this function is copied directly from KromUtils.
//Do not remove and add KromUtils to uses, that would cause the Linux build to fail
function TimeGet: Cardinal;
begin
  {$IFDEF MSWindows}
  Result := TimeGetTime; //Return milliseconds with ~1ms precision
  {$ENDIF}
  {$IFDEF Unix}
  Result := Cardinal(Trunc(Now * 24 * 60 * 60 * 1000) mod high(Cardinal));
  {$ENDIF}
end;


function GetTimeSince(aTime: Cardinal): Cardinal;
begin
  //TimeGet will loop back to zero after ~49 days since system start
  Result := (Int64(TimeGet) - Int64(aTime) + Int64(High(Cardinal))) mod Int64(High(Cardinal));
end;


function MapSizeText(X,Y: Word): string;
begin
  case X * Y of
            1.. 48* 48: Result := 'XS';
     48* 48+1.. 80* 80: Result := 'S';
     80* 80+1..128*128: Result := 'M';
    128*128+1..176*176: Result := 'L';
    176*176+1..224*224: Result := 'XL';
    224*224+1..320*320: Result := 'XXL';
    else                Result := '???';
  end;
end;


function KMGetCursorDirection(X,Y: Integer): TKMDirection;
var Ang, Dist: Single;
begin
  Dist := GetLength(X, Y);
  if Dist > DirCursorNARadius then
  begin
    //Convert XY to angle value
    Ang := ArcTan2(Y/Dist, X/Dist) / Pi * 180;

    //Convert angle value to direction
    Result := TKMDirection((Round(Ang + 270 + 22.5) mod 360) div 45 + 1);
  end
  else
    Result := dir_NA;
end;


{Returns point where unit should be placed regarding direction & offset from Commanders position}
// 23145     231456
// 6789X     789xxx
function GetPositionInGroup2(OriginX, OriginY: Word; aDir: TKMDirection; aIndex, aUnitPerRow: Word; MapX, MapY: Word; out aTargetCanBeReached: Boolean): TKMPoint;
const
  DirAngle: array [TKMDirection] of Word   = (0, 0, 45, 90, 135, 180, 225, 270, 315);
  DirRatio: array [TKMDirection] of Single = (0, 1, 1.41, 1, 1.41, 1, 1.41, 1, 1.41);
var
  PlaceX, PlaceY, ResultX, ResultY: integer;
begin
  Assert(aUnitPerRow > 0);
  if aIndex = 0 then
  begin
    ResultX := OriginX;
    ResultY := OriginY;
  end
  else
  begin
    if aIndex <= aUnitPerRow div 2 then
      Dec(aIndex);
    PlaceX := aIndex mod aUnitPerRow - aUnitPerRow div 2;
    PlaceY := aIndex div aUnitPerRow;

    ResultX := OriginX + Round( PlaceX*DirRatio[aDir]*cos(DirAngle[aDir]/180*pi) - PlaceY*DirRatio[aDir]*sin(DirAngle[aDir]/180*pi) );
    ResultY := OriginY + Round( PlaceX*DirRatio[aDir]*sin(DirAngle[aDir]/180*pi) + PlaceY*DirRatio[aDir]*cos(DirAngle[aDir]/180*pi) );
  end;

  aTargetCanBeReached := InRange(ResultX, 1, MapX-1) and InRange(ResultY, 1, MapY-1);
  //Fit to bounds
  Result.X := EnsureRange(ResultX, 1, MapX-1);
  Result.Y := EnsureRange(ResultY, 1, MapY-1);
end;


//See Docs\GetPositionFromIndex.xls for explanation
function GetPositionFromIndex(aOrigin: TKMPoint; aIndex: Byte): TKMPointI;
const
  Rings: array[1..10] of Word =
//Ring#  1  2  3  4   5   6   7    8    9    10
        (0, 1, 9, 25, 49, 81, 121, 169, 225, 289);
var
  Ring, Span, Span2, Orig: Byte;
  Off1,Off2,Off3,Off4,Off5: Byte;
begin
  //Quick solution
  if aIndex = 0 then
  begin
    Result.X := aOrigin.X;
    Result.Y := aOrigin.Y;
    Exit;
  end;

  //Find ring in which Index is located
  Ring := 0;
  repeat inc(Ring); until(Rings[Ring]>aIndex);
  dec(Ring);

  //Remember Ring span and half-span
  Span := Ring*2-1-1; //Span-1
  Span2 := Ring-1;    //Half a span -1

  //Find offset from Rings 1st item
  Orig := aIndex - Rings[Ring];

  //Find Offset values in each span
  Off1 := min(Orig,Span2); dec(Orig,Off1);
  Off2 := min(Orig,Span);  dec(Orig,Off2);
  Off3 := min(Orig,Span);  dec(Orig,Off3);
  Off4 := min(Orig,Span);  dec(Orig,Off4);
  Off5 := min(Orig,Span2-1); //dec(Orig,Off5);

  //Compute result
  Result.X := aOrigin.X + Off1 - Off3 + Off5;
  Result.Y := aOrigin.Y - Span2 + Off2 - Off4;
end;


//Use this function to convert platform-specific path delimiters
function FixDelim(const aString: string): string;
begin
  Result := StringReplace(aString, '\', PathDelim, [rfReplaceAll, rfIgnoreCase]);
end;


function GetPingColor(aPing: Word): Cardinal;
begin
  case aPing of
    0..299  : Result := icGreen;
    300..599: Result := icYellow;
    600..999: Result := icOrange;
    else      Result := icRed;
  end;
end;


function GetFPSColor(aFPS: Word): Cardinal;
begin
  case aFPS of
    0..9  : Result := icRed;
    10..12: Result := icOrange;
    13..15: Result := icYellow;
    else    Result := icGreen;
  end;
end;


procedure ConvertRGB2HSB(aR, aG, aB: Integer; out oH, oS, oB: Single);
var
  R, G, B: Single;
  Rdlt, Gdlt, Bdlt, Vmin, Vmax, Vdlt: Single;
begin
  R := aR / 255;
  G := aG / 255;
  B := aB / 255;

  Vmin := min(R, min(G, B));
  Vmax := max(R, max(G, B));
  Vdlt := Vmax - Vmin;
  oB := (Vmax + Vmin) / 2;
  if Vdlt = 0 then
  begin
    oH := 0.5;
    oS := 0;
  end
  else
  begin // Middle of HSImage
    if oB < 0.5 then
      oS := Vdlt / (Vmax + Vmin)
    else
      oS := Vdlt / (2 - Vmax - Vmin);

    Rdlt := (R - Vmin) / Vdlt;
    Gdlt := (G - Vmin) / Vdlt;
    Bdlt := (B - Vmin) / Vdlt;

    if R = Vmax then oH := (Gdlt - Bdlt) / 6 else
    if G = Vmax then oH := 1/3 - (Rdlt - Bdlt) / 6 else
    if B = Vmax then oH := 2/3 - (Gdlt - Rdlt) / 6 else
                      oH := 0;

    if oH < 0 then oH := oH + 1;
    if oH > 1 then oH := oH - 1;
  end;
end;


procedure ConvertHSB2RGB(aHue, aSat, aBri: Single; out R, G, B: Byte);
const V = 6;
var Hue, Sat, Bri, Rt, Gt, Bt: Single;
begin
  Hue := EnsureRange(aHue, 0, 1);
  Sat := EnsureRange(aSat, 0, 1);
  Bri := EnsureRange(aBri, 0, 1);

  //Hue
  if Hue < 1/6 then
  begin
    Rt := 1;
    Gt := Hue * V;
    Bt := 0;
  end else
  if Hue < 2/6 then
  begin
    Rt := (2/6 - Hue) * V;
    Gt := 1;
    Bt := 0;
  end else
  if Hue < 3/6 then
  begin
    Rt := 0;
    Gt := 1;
    Bt := (Hue - 2/6) * V;
  end else
  if Hue < 4/6 then
  begin
    Rt := 0;
    Gt := (4/6 - Hue) * V;
    Bt := 1;
  end else
  if Hue < 5/6 then
  begin
    Rt := (Hue - 4/6) * V;
    Gt := 0;
    Bt := 1;
  end else
  //if Hue < 6/6 then
  begin
    Rt := 1;
    Gt := 0;
    Bt := (6/6 - Hue) * V;
  end;

  //Saturation
  Rt := Rt + (0.5 - Rt) * (1 - Sat);
  Gt := Gt + (0.5 - Gt) * (1 - Sat);
  Bt := Bt + (0.5 - Bt) * (1 - Sat);

  //Brightness
  if Bri > 0.5 then
  begin
    //Mix with white
    Rt := Rt + (1 - Rt) * (Bri - 0.5) * 2;
    Gt := Gt + (1 - Gt) * (Bri - 0.5) * 2;
    Bt := Bt + (1 - Bt) * (Bri - 0.5) * 2;
  end
  else if Bri < 0.5 then
  begin
    //Mix with black
    Rt := Rt * (Bri * 2);
    Gt := Gt * (Bri * 2);
    Bt := Bt * (Bri * 2);
  end;
  //if Bri = 127 then color remains the same

  R := Round(Rt * 255);
  G := Round(Gt * 255);
  B := Round(Bt * 255);
end;


function ApplyBrightness(aColor: Cardinal; aBrightness: Byte): Cardinal;
begin
  Result := Round((aColor and $FF) / 255 * aBrightness)
            or
            Round((aColor shr 8 and $FF) / 255 * aBrightness) shl 8
            or
            Round((aColor shr 16 and $FF) / 255 * aBrightness) shl 16
            or
            (aColor and $FF000000);
end;


//Desaturate and lighten the color best done in HSB colorspace
function FlagColorToTextColor(aColor: Cardinal): Cardinal;
var
  R, G, B: Byte;
  Hue, Sat, Bri: Single;
begin
  ConvertRGB2HSB(aColor and $FF, aColor shr 8 and $FF, aColor shr 16 and $FF, Hue, Sat, Bri);

  //Desaturate and lighten
  Sat := Min(Sat, 0.93);
  Bri := Max(Bri + 0.1, 0.2);
  ConvertHSB2RGB(Hue, Sat, Bri, R, G, B);

  //Preserve transparency value
  Result := (R + G shl 8 + B shl 16) or (aColor and $FF000000);
end;


//Convert DateTime to string xx:xx:xx where hours have at least 2 digits
//F.e. we can have 72:12:34 for 3 days long game
function TimeToString(aTime: TDateTime): string;
begin
  //We can't use simple Trunc(aTime * 24 * 60 * 60) maths because it is prone to rounding errors
  //e.g. 3599 equals to 59:58 and 3600 equals to 59:59
  //That is why we resort to DateUtils routines which are slower but much more correct
  Result :=  Format('%.2d', [HoursBetween(aTime, 0)]) + FormatDateTime(':nn:ss', aTime);
end;


//Make a string wrapped into color code
function WrapColor(aText: UnicodeString; aColor: Cardinal): UnicodeString;
begin
  Result := '[$' + IntToHex(aColor and $00FFFFFF, 6) + ']' + aText + '[]';
end;


function WrapColorA(aText: AnsiString; aColor: Cardinal): AnsiString;
begin
  Result := '[$' + IntToHex(aColor and $00FFFFFF, 6) + ']' + aText + '[]';
end;


function StripColor(aText: UnicodeString): UnicodeString;
var
  I: Integer;
  skippingMarkup: Boolean;
begin
  Result := '';
  skippingMarkup := False;

  for I := 1 to Length(aText) do
  begin
    if (I+1 <= Length(aText))
    and ((aText[I] + aText[I+1] = '[$') or (aText[I] + aText[I+1] = '[]')) then
      skippingMarkup := True;

    if not skippingMarkup then
      Result := Result + aText[I];

    if skippingMarkup and (aText[I] = ']') then
      skippingMarkup := False;
  end;
end;


//Taken from: http://delphi.about.com/od/adptips2005/qt/parsedelimited.htm
procedure ParseDelimited(const SL: TStringList; const Value: string; const Delimiter: string);
var
  dx: integer;
  ns: string;
  txt: string;
  Delta: integer;
begin
  Delta := Length(Delimiter);
  txt := Value + Delimiter;
  SL.BeginUpdate;
  SL.Clear;
  try
    while Length(txt) > 0 do
    begin
      dx := Pos(Delimiter, txt);
      ns := Copy(txt, 0, dx-1);
      SL.Add(ns);
      txt := Copy(txt, dx+Delta, MaxInt);
    end;
  finally
    SL.EndUpdate;
  end;
end;


//Quote from page 5 of 'Random Number Generators': "We recommend the construction of an initialization procedure,
//Randomize, which prompts for an initial value of seed and forces it to be an integer between 1 and 2^31 - 2."
procedure SetKaMSeed(aSeed: Integer);
begin
  Assert(InRange(aSeed,1,2147483646),'KaMSeed initialised incorrectly: '+IntToStr(aSeed));
  if CUSTOM_RANDOM then
    fKaMSeed := aSeed
  else
    RandSeed := aSeed;
end;


function GetKaMSeed:integer;
begin
  if CUSTOM_RANDOM then
    Result := fKaMSeed
  else
    Result := RandSeed;
end;


//Taken from "Random Number Generators" by Stephen K. Park and Keith W. Miller.
(*  Integer  Version  2  *)
function KaMRandom: Extended;
const
  A = 16807;
  M = 2147483647; //Prime number 2^31 - 1
  Q = 127773; // M div A
  R = 2836; // M mod A
var
  C1, C2, NextSeed: integer;
begin
  if not CUSTOM_RANDOM then
  begin
    Result := Random;
    Exit;
  end;

  Assert(InRange(fKaMSeed,1,M-1), 'KaMSeed initialised incorrectly: '+IntToStr(fKaMSeed));
  C2 := fKaMSeed div Q;
  C1 := fKaMSeed mod Q;
  NextSeed := A*C1 - R*C2;

  if NextSeed > 0 then
    fKaMSeed := NextSeed
  else
    fKaMSeed := NextSeed + M;

  Result := fKaMSeed / M;
end;


function KaMRandom(aMax:integer):integer;
begin
  if CUSTOM_RANDOM then
    Result := trunc(KaMRandom*aMax)
  else
    Result := Random(aMax);
end;


function KaMRandomS(Range_Both_Directions:integer):integer; overload;
begin
  Result := KaMRandom(Range_Both_Directions*2+1)-Range_Both_Directions;
end;


function KaMRandomS(Range_Both_Directions:single):single; overload;
begin
  Result := KaMRandom(round(Range_Both_Directions*20000)+1)/10000-Range_Both_Directions;
end;


end.
